/******************************************************************************
 Copyright (C), 2014, TP-LINK TECHNOLOGIES CO., LTD.  All rights reserved.

 File name	: isGuestNetworkSta.c
 Version	: v1.0
 Description: judge a given station is associate with our guest network.
 			  for BCM project only.

 Author		: Jiangji
 Create Date: 2014/02/17

 History: 
------------------------------------------------------------
 01, 17Feb14, jiangji, create this file.
***********************************************************/

#include <stdio.h>
#include <stdlib.h> /* strtoul */
#include <string.h> /* strcpy */
#include <limits.h> /* for ULONG_MAX*/
#include <arpa/inet.h> /* for INADDR_NONE */
#include "wlanIoctl.h"

/* #define _IGNS_DEBUG_ */
#ifdef _IGNS_DEBUG_
#define igns_print printf
#else
#define igns_print(fmt, ...)
#endif

#define PATH_PROCNET_ARP		"/proc/net/arp"
#define MAC_ADDRESS_LENGTH		6
#define LINE_BUF_LEN			200
#define ITEM_BUF_LEN			100

#define WLAN_IFNAME_LEN			16

/* return value define */
enum
{
	IGNS_RT_ERROR = -2,		/* ERROR detection */
	IGNS_RT_NA = -1,		/* Can't find the sta's macaddr, and couldn't make judgment */
	IGNS_RT_FALSE = 0,		/* The sta isn't a guest network client */
	IGNS_RT_TRUE = 1		/* The sta is a guest network client */
};

enum
{
	WLAN_BAND_2G = 0,
	WLAN_BAND_5G,
	WLAN_MAX_BAND_NUM
};

/* iterates arp cache table to find the @ipaddr, return its @macaddr to caller */
static int findMacByIp(const unsigned long ipaddr, unsigned char *macaddr)
{
	FILE *fp = NULL;
	char line[LINE_BUF_LEN] = {'\0'};
	char ip[ITEM_BUF_LEN] = {'\0'};
	char hw[ITEM_BUF_LEN] = {'\0'};
	char mask[ITEM_BUF_LEN] = {'\0'};
	char dev[ITEM_BUF_LEN] = {'\0'};
	int type = 0;
	int flags = 0;

	int num = 0;
	unsigned long cacheIp = 0;

	int ret = -1;

	/* open kernel's arp cache table through proc filesystem */
	if ((fp = fopen(PATH_PROCNET_ARP, "r")) == NULL)
	{
		return ret;
	}

	/* Bypass header */
	if (fgets(line, LINE_BUF_LEN, fp) != (char *)NULL)
	{
		/* Read arp cache entry */
		for (; fgets(line, LINE_BUF_LEN, fp);)
		{
			num = sscanf(line, "%s 0x%x 0x%x %100s %100s %100s\n",
				ip, &type, &flags, hw, mask, dev);
			if (num < 4)
			{
				break;
			}

			cacheIp = inet_addr(ip);
			if (INADDR_NONE == cacheIp)
			{
				continue;
			}

			/* find the target ip */
			if (cacheIp == ipaddr)
			{
				ret = sscanf(hw, "%x:%x:%x:%x:%x:%x", (macaddr + 0), (macaddr + 1), (macaddr + 2),
					(macaddr + 3), (macaddr + 4), (macaddr + 5));
				if (ret == EOF)
				{
					break;
				}

				ret = 0;
				break;
			}
		}
	}

	fclose(fp);
	return ret;
}

/* whether guest network is enabled on @band */
static BOOL isGuestNetworkEnabled(const int band)
{
	char wifName[WLAN_IFNAME_LEN] = {'\0'};
	int ret = 0;

	/* first: check wlx.1 is exist */
	ret = sprintf(wifName, "wl%d.1", band);
	if( ret <= 0)
	{
		return FALSE;
	}

	if (wl_check(wifName))
	{
		igns_print("%s doesn't exist!\r\n", wifName);
		return FALSE;
	}

	/* second: WDS & GN cann't work concurrently, WDS open means GN close */
	ret = sprintf(wifName, "eth%d", (band + 1));
	if (ret < 0)
	{
		return FALSE;
	}

	if (!wl_bridge(wifName))
	{
		igns_print("bridge is enabled on band %d!\r\n", band);
		return FALSE;
	}

	/* last: check ethx is up */
	if (wl_bssup(wifName))
	{		
		igns_print("%s is down!\r\n", wifName);
		return FALSE;
	}

	return TRUE;
}

/* find @macaddr in @band assoclist */
static BOOL findStaInGuestNetwork(const int band, const unsigned char *macaddr)
{
	BOOL ret = FALSE;
	char wifName[WLAN_IFNAME_LEN] = {'\0'};
	unsigned char staListBuf[WLAN_MAX_STA_NUM * (MAC_ADDRESS_LENGTH + 1)] = {'\0'};
	unsigned char *pStaMac = NULL;
	unsigned int staNum = 0;
	unsigned int i = 0;

	memset(staListBuf, '\0', sizeof(staListBuf));

	/* wlx.1 is guest network interface */
	if (sprintf(wifName, "wl%d.1", band) <= 0)
	{
		return ret;
	}

	if(wl_assoclist(wifName, staListBuf, sizeof(staListBuf)))
	{
		return ret;
	}

	staNum = *((unsigned int *)staListBuf);
	pStaMac = staListBuf + sizeof(unsigned int);

	igns_print("%d clients assoc on %s.\r\n", staNum, wifName);

	/* find @macaddr in assoclist */
	for(i = 0; i < staNum; i++)
	{
		pStaMac += (i * MAC_ADDRESS_LENGTH);
		igns_print("client%2d: %02x:%02x:%02x:%02x:%02x:%02x\n", (i + 1), pStaMac[0], pStaMac[1],
			pStaMac[2], pStaMac[3], pStaMac[4], pStaMac[5]);
		if (memcmp(pStaMac, macaddr, MAC_ADDRESS_LENGTH) == 0)
		{
			ret = TRUE;
			break;
		}
	}
	
	return ret;
}


/*
 * usage:
 *		./isGuestNetworkSta ip_addr
 *		
 * 		@ip_addr must be formated in Net Bit Order
 */
int main(int argc, char **argv)
{
	int ret = IGNS_RT_ERROR;
	int bandIdx = WLAN_BAND_2G;
	unsigned long staIp = 0;
	unsigned char staMac[MAC_ADDRESS_LENGTH + 1] = {'\0'};

	/* get the client's ip addr from argv */
	if (argc == 2)
	{
		staIp = strtoul(argv[1], NULL, 10);
		if (staIp <= 0 || staIp > ULONG_MAX)
		{
			igns_print("bad ip input!\r\n");
			ret = IGNS_RT_ERROR;
			goto done;
		}
	}

	/* map client's ip to mac */

	if (findMacByIp(staIp, staMac) != 0)
	{
		char cmdline[100] = {'\0'};
		sprintf(cmdline, "arping -q -I br0 -c 1 %s &", inet_ntoa(*((struct in_addr *)&staIp)));
		system(cmdline);

		igns_print("trigger arping to refresh arp cache.\r\n");
		ret = IGNS_RT_NA;
		
		goto done;
	}

	igns_print("mapping %s -> %02x:%02x:%02x:%02x:%02x:%02x.\r\n", inet_ntoa(*((struct in_addr *)&staIp)),
		staMac[0], staMac[1], staMac[2], staMac[3], staMac[4], staMac[5]);

	ret = IGNS_RT_FALSE;

	/* find client's mac in each band */
	for (bandIdx = WLAN_BAND_2G; bandIdx < WLAN_MAX_BAND_NUM; bandIdx++)
	{
		/* check Guest Network is enabled */
		if (isGuestNetworkEnabled(bandIdx) == FALSE)
		{
			igns_print("guest network is close on %s.\r\n", bandIdx?"5g":"2g");
			continue;
		}

		if (findStaInGuestNetwork(bandIdx, staMac) == TRUE)
		{
			igns_print("find client on %s.\r\n", bandIdx?"5g":"2g");
			ret = IGNS_RT_TRUE;
			break;
		}
	}

	igns_print("return value is %d.\r\n", ret);
done:	
	return ret;
}

